import fs from 'fs';
import path from 'path';
import matter from 'gray-matter';
import { remark } from 'remark';
import html from 'remark-html';

const postsDirectory = path.join(process.cwd(), 'content/blog');

/**
 * Obtiene todos los slugs de los posts
 * @returns {string[]} Array de slugs
 */
export function getAllPostSlugs() {
  if (!fs.existsSync(postsDirectory)) {
    return [];
  }
  const fileNames = fs.readdirSync(postsDirectory);
  return fileNames
    .filter(fileName => fileName.endsWith('.md'))
    .map(fileName => fileName.replace(/\.md$/, ''));
}

/**
 * Obtiene metadatos de todos los posts (sin contenido)
 * Ordenados por fecha descendente
 * @returns {Array<{slug, title, excerpt, date, image}>}
 */
export function getAllPostsMetadata() {
  const slugs = getAllPostSlugs();

  const posts = slugs.map(slug => {
    const fullPath = path.join(postsDirectory, `${slug}.md`);
    const fileContents = fs.readFileSync(fullPath, 'utf8');
    const { data } = matter(fileContents);

    return {
      slug,
      title: data.title,
      excerpt: data.excerpt,
      date: data.date,
      image: data.image,
    };
  });

  // Ordenar por fecha descendente
  return posts.sort((a, b) => (a.date < b.date ? 1 : -1));
}

/**
 * Obtiene un post completo por slug (con contenido HTML)
 * @param {string} slug
 * @returns {Promise<{slug, title, excerpt, date, image, content}>}
 */
export async function getPostBySlug(slug) {
  const fullPath = path.join(postsDirectory, `${slug}.md`);

  if (!fs.existsSync(fullPath)) {
    return null;
  }

  const fileContents = fs.readFileSync(fullPath, 'utf8');
  const { data, content } = matter(fileContents);

  // Convertir Markdown a HTML
  const processedContent = await remark()
    .use(html, { sanitize: false })
    .process(content);

  const contentHtml = processedContent.toString();

  return {
    slug,
    title: data.title,
    excerpt: data.excerpt,
    date: data.date,
    image: data.image,
    content: contentHtml,
  };
}

/**
 * Obtiene los N posts mas recientes (para home)
 * @param {number} count
 * @returns {Array}
 */
export function getRecentPosts(count = 3) {
  return getAllPostsMetadata().slice(0, count);
}
